<?php

trait GalaxPayGetInfoOrder
{

    public $qtdSubscription, $hasSubscription = false, $hasProduct = false, $feeValue = 0;

    protected function getCurrentUser()
    {

        return wp_get_current_user();
    }

    protected function getUserDocumentAndName()
    {

        if ($this->order->get_meta('_billing_persontype') == '2') {

            $name = $this->order->get_billing_company();

            $cpf_or_cnpj = $this->order->get_meta('_billing_cnpj');
        } else {

            $name = $this->order->get_billing_first_name() . ' ' . $this->order->get_billing_last_name();

            $cpf_or_cnpj = $this->order->get_meta('_billing_cpf');
        }

        return array('name' => $name, 'document' => $cpf_or_cnpj);
    }

    private function getOnlyNumbers($doc)
    {

        return preg_replace('/\D/', '', $doc);
    }

    protected function getArrayCustomer()
    {

        $currentUser = $this->getCurrentUser();

        $userDocumentAndName = $this->getUserDocumentAndName();

        $user_id = $currentUser->ID;

        $user_code = get_user_meta($user_id, 'gpay_user_code', true);

        if ($user_code == false) {

            $user_code = 'wc-' . $user_id . '-' . time();

            add_user_meta($user_id, 'gpay_user_code', $user_code, true);
        }

        $phones = [
            $this->getOnlyNumbers($this->order->get_billing_phone()),
            $this->getOnlyNumbers($this->getCelphone())
        ];

        return array(
            'myId' => $user_id,
            'name' => $userDocumentAndName['name'],
            'emails' => [$this->order->get_billing_email()],
            'document' => $this->getOnlyNumbers($userDocumentAndName['document']),
            'Address' => [
                'street' => $this->order->get_billing_address_1(),
                'number' => $this->order->get_meta('_billing_number'),
                'complement' => $this->order->get_billing_address_2(),
                'zipCode' => $this->getOnlyNumbers($this->order->get_billing_postcode()),
                'neighborhood' => $this->order->get_meta('_billing_neighborhood'),
                'city' => $this->order->get_billing_city(),
                'state' => $this->order->get_billing_state(),
            ],
            'phones' => array_filter($phones),
        );
    }

    protected function getCelphone()
    {

        $cellPhone = sanitize_text_field($_POST['billing_cellphone']);

        return $cellPhone;
    }

    public function processProducts()
    {

        $this->hasSubscription = false;

        $this->hasProduct = false;

        $items = $this->order->get_items();

        foreach ($items as $item) {

            $product = $this->order->get_product_from_item($item);

            $this->qtdSubscription = $product->get_meta('_subscription_length');

            if ($this->isTypeSubscription($product)) {
                $this->feeValue = $product->subscription_sign_up_fee;
                if ($this->hasSubscription) {

                    return $this->abort('Só é permitido fazer checkout de uma assinatura por vez!');
                }

                $this->hasSubscription = true;
            } else {

                $this->hasProduct = true;
            }
        }
    }

    public static function hasSubscriptionAdditionalPlugin()
    {
        //extension codup_build_your_own_subscription 

        $data = unserialize(base64_decode(sanitize_text_field(wp_unslash($_COOKIE['wp-subscription_fields']))));
        if (!empty($data['billing_interval']) && $_COOKIE['wp-is_subscribed'] == '1') {
            return true;
        }
        return false;
    }

    public function getOrderType()
    {

        $this->processProducts();

        if (self::hasSubscriptionAdditionalPlugin()) {

            return 'subscription';
        }



        if ($this->hasProduct && $this->hasSubscription) {

            $this->abort('Não é possível fazer pagamento de produtos comuns junto com alguma assinatura.');
        }

        if ($this->hasProduct) {

            return 'single';
        }

        if ($this->hasSubscription) {

            return 'subscription';
        }
    }

    public function isSubscription()
    {

        return $this->getOrderType() == 'subscription';
    }

    public function getSubscription()
    {
        if ($this->isSubscription()) {
            $wcSubscriptions = wcs_get_subscriptions_for_order($this->order->id);
            return end($wcSubscriptions);
        }
        return null;
    }

    protected function isTypeSubscription(WC_Product $product)
    {

        return (bool) preg_match('/subscription/', $product->get_type());
    }

    protected function validateValue()
    {

        if ($this->order->total < 5) {

            $message = 'O valor mínimo para gerar um boleto é R$ 5,00';

            wc_add_notice($message, 'error');

            throw new Exception($message);
        }
    }

    protected function checkInstallments()
    {

        $installments = $this->getInstallments();

        if (!empty($installments)) {

            if ($installments != 1) {

                $this->abort('Não é possível parcelar uma assinatura.');
            }
        }
    }

    // protected function isRenewalOrder()
    // {
    //   $subscriptionsNew = wcs_get_subscriptions_for_order($this->order);
    //   if (empty($subscriptionsNew)) {
    //     $subscriptionsRenew = wcs_get_subscriptions_for_renewal_order($this->order);
    //     if (empty($subscriptionsRenew)) {
    //       return false;
    //     }
    //     return true;
    //   }
    //   return false;
    // }





    protected function isBetween($valueToCompare, $minValue, $maxValue)
    {

        if ($valueToCompare >= $minValue) {

            if ($valueToCompare <= $maxValue) {

                return true;
            }
        }

        return false;
    }

    protected function installments()
    {

        $numberInstallments = sanitize_text_field($_POST['installmentsNumber']);

        if (empty($numberInstallments)) {

            return 1;
        }

        return $numberInstallments;
    }

    protected function getCardNumber()
    {

        $cardNumber = sanitize_text_field($_POST['cardNumber']);

        if ($cardNumber) {

            return $cardNumber;
        }

        $message = 'Número de cartão inválido!';

        $this->abort(__($message, GALAX_IDENTIFIER), true);
    }

    protected function getCardHolder()
    {

        $holder = sanitize_text_field($_POST['cardHolder']);

        if (is_string($holder)) {

            return $holder;
        }

        $message = 'Nome inválido!';

        $this->abort(__($message, GALAX_IDENTIFIER), true);
    }

    public function getCardMonthExpiry()
    {

        $cardMonthExpiry = sanitize_text_field($_POST['cardMonthExpiry']);

        if ($cardMonthExpiry) {

            return $cardMonthExpiry;
        }

        $message = 'Mês de expiração inválido!';

        $this->abort(__($message, GALAX_IDENTIFIER), true);
    }

    protected function getCardYearExpiry()
    {

        $cardYearExpiry = sanitize_text_field($_POST['cardYearExpiry']);

        if ($cardYearExpiry) {

            return $cardYearExpiry;
        }

        $message = 'Ano de expiração inválido!';

        $this->abort(__($message, GALAX_IDENTIFIER), true);
    }

    protected function getCvv()
    {

        $cardCvv = sanitize_text_field($_POST['cardCvv']);

        if ($cardCvv) {

            return $cardCvv;
        }

        $message = 'Número de CVV inválido!';

        $this->abort(__($message, GALAX_IDENTIFIER), true);
    }

    protected function getPaymentMethodCreditCard()
    {

        return [
            'qtdInstallments' => $this->getInstallments(),
            'Card' => [
                'number' => $this->getCardNumber(),
                'holder' => $this->getCardHolder(),
                'expiresAt' => $this->getCardYearExpiry() . '-' . $this->getCardMonthExpiry(),
                'cvv' => $this->getCvv(),
            ]
        ];
    }

    protected function getPaymentMethodBoleto()
    {

        return [
            'deadlineDays' => $this->container->getQuantityDaysBeforeSale(),
            'instructions' => $this->container->getAdditionalInfoBoleto() . ' - Pedido Código ' . $this->order->id,
        ];
    }

    protected function getPaymentMethodPix()
    {

        if ($this->isSubscription()) {
            return [
                "Deadline" => [
                    "type" => $this->container->getTypeDeadlinePixSubs(),
                    "value" => $this->container->getTimeDeadlinePixSubs()
                ],
                'instructions' => 'Pedido Código ' . $this->order->id
            ];
        } else {
            return [
                "Deadline" => [
                    "type" => $this->container->getTypeDeadlinePix(),
                    "value" => $this->container->getTimeDeadlinePix()
                ],
                'instructions' => 'Pedido Código ' . $this->order->id
            ];
        }
    }

    protected function getInstallments()
    {

        $numberInstallments = sanitize_text_field($_POST['installmentsNumber']);

        if (empty($numberInstallments)) {

            return 1;
        }

        return $numberInstallments;
    }

    protected function getCommonBody($type)
    {
        $fieldPayDay = $type == 'charge' ? 'payday' : 'firstPayDayDate';
        if ($this->getSubscription()) {
            $total = $this->getSubscription()->total;
        } else {
            $total = $this->order->total;
        }
        
        $body = [
            'Customer' => $this->getArrayCustomer(),
            'myId' => $this->order->id,
            $fieldPayDay => $this->getPayday($type),
            'value' => (int) $this->getOnlyNumbers($total),
        ];

        $types = [
            'card' => ['creditcard', 'PaymentMethodCreditCard'],
            'boleto' => ['boleto', 'PaymentMethodBoleto'],
            'pix' => ['pix', 'PaymentMethodPix']
        ];
        if (!isset($types[$this->typePayment])) {
            $this->container->logger->log(sprintf('Erro no pagamento do pedido.', $this->order->id));
            $message = sprintf(__('Tipo de pagamento não encontrado.', GALAX_IDENTIFIER), $this->container->api->last_error);
            $this->order->update_status('failed', $message);
            new Exception($message);
        }
        $config = $types[$this->typePayment];
        $body['mainPaymentMethodId'] = $config['0'];
        $method = 'get' . $config['1'];
        $body[$config['1']] = $this->$method();

        return $body;
    }

    private function getPayday($type)
    {
        if ($type == 'subscription') {
            $wcSubscription = $this->getSubscription();
            if ($wcSubscription->trial_period <> '') {
                return date('Y-m-d', $wcSubscription->get_time('trial_end'));
            }
        }
        return date('Y-m-d');
    }

    public function getQuantitySubscription()
    {

        $this->processProducts();

        $data = unserialize(base64_decode(sanitize_text_field(wp_unslash($_COOKIE['wp-subscription_fields']))));

        if (!empty($data['billing_interval'])) {

            return '0'; //indeterminada
        }



        if ($this->qtdSubscription == null) {

            $this->qtdSubscription = 0;
        }

        return $this->qtdSubscription;
    }

    protected function getPeriodicityData($wcSubscription)
    {
        if (self::hasSubscriptionAdditionalPlugin()) {
            return $this->getPeriodicityDataByPlugin();
        }

        $daysToCalcArray = array(
            'day' => 1,
            'week' => 7,
            'month' => 30,
            'year' => 365
        );

        $finalDate = $daysToCalcArray[$wcSubscription->billing_period] * $wcSubscription->billing_interval;
        if ($this->isBetween($finalDate, 5, 9)) {
            return 'weekly';
        }

        if ($this->isBetween($finalDate, 12, 16)) {
            return 'biweekly';
        }

        if ($this->isBetween($finalDate, 26, 33)) {
            return 'monthly';
        }

        if ($this->isBetween($finalDate, 56, 64)) {
            return 'bimonthly';
        }

        if ($this->isBetween($finalDate, 84, 94)) {
            return 'quarterly';
        }

        if ($this->isBetween($finalDate, 168, 184)) {
            return 'biannual';
        }

        if ($this->isBetween($finalDate, 362, 369)) {
            return 'yearly';
        }

        $this->abort('Periodicidade está incorreta. Entre em contato com o administrador da loja.', true);
    }

    private function getTransactionsSubscription()
    {
        $return = [
            [
                'installment' => 1,
                'myId' => $this->order->id
            ]
        ];
        $wcSubscription = $this->getSubscription();        
        if ($wcSubscription && $this->order->total <> $wcSubscription->total) {
            $valueFirstTransaction = $this->order->total;
            if ($wcSubscription->schedule_trial_end > 0) {
                $valueFirstTransaction += $wcSubscription->total;
            }
            $valueFirstTransaction = number_format($valueFirstTransaction, 2);
            $return[0]['value'] = (int) $this->getOnlyNumbers($valueFirstTransaction);
        }
        return $return;
    }

    private function getPeriodicityDataByPlugin()
    {
        $data = unserialize(base64_decode(sanitize_text_field(wp_unslash($_COOKIE['wp-subscription_fields']))));
        if (!empty($data['billing_interval'])) {
            $interval = $data['billing_interval'];
            
            if ($data['billing_period'] == 'week') {
                $valoresPossiveis = [
                    1 => 'weekly',
                    2 => 'biweekly',
                ];
            } else if ($data['billing_period'] == 'month') {
                $valoresPossiveis = [
                    1 => 'monthly',
                    2 => 'bimonthly',
                    3 => 'quarterly'
                ];
            } else if ($data['billing_period'] == 'yearly') {
                $valoresPossiveis = [
                    1 => 'yearly',
                    2 => 'biannual'
                ];
            }
            
            if (isset($valoresPossiveis[$interval])) {
                return $valoresPossiveis[$interval];
            } else {
                $this->abort('Periodicidade está incorreta. Entre em contato com o administrador da loja.', true);
            }
        }
    }

}
